/*!
 * Copyright (c) 2005, Freescale Semiconductor
 *
 * Freescale Confidential Proprietary
 * \file    mcu_spi_config.c
 * \brief   This is the SMAC C source for the SPI driver
 * \author  Larry Roshak
 * \version 4.1a
 * \date    2005/08/11 20:44:37 
 * 
 * \b Description:
 *
 * This is the SMAC C source driver file for the 
 * The driver control interface to the SPI.
 *
 * \b Department: Freescale Radio Products Division 
 *
 * \b Project: SMAC (Simple Media Access Controller)
 *
 * \b History:
 * - 17/04/2004 : Initial Development. (FLR005)
 * - 07/01/2005 : Code Standarization. (A19259)
 * - 11/16/2005 Doc. update to Doxygen compliant by Gonzalo Delgado 
 * Huitron rgd04c
 */ 

#include "smac_pub_def.h"
#include "smac_mcu_hw_config.h"
#include "smac_drivers.h"
#include "smac_MC13192_hw_config.h"
#include "smac_MC13192_regs.h"
#include "smac_mcu_spi_config.h"
#include "mcf5271.h"
#include "init5270.h"
#include "mcf5271_qspi.h"
#include "mcf5271_gpio.h"
#include "hardware.h"

/*!
  \fn void SMAC_SPIInit(void)
  \brief Initialize the SPI Control and Baud Rate Registers
*/
void SMAC_SPIInit(void)
{
	QSPI_5271_Regs *qspi = get_qspi_ptr();
	GPIO_5271_Regs *gpio = get_gpio_ptr();
  UINT8 clockDiv;

  // Setup the pin assignment register in the GPIO module for QSPI

  // MNT - 4/5/2007 -  Clean this up later - this seems to work
  // MNT - 4/6/2007 - Change the qspi cs0 to a gpio
	//gpio->par_qspi		= 0x3F;	
	gpio->par_qspi		= 0x1F;	
  // MNT - 4/6/2007 - Configure Bit 3 as an output - Table 12-1 in the RM
	gpio->pddr_qspi		= 0x08; 	
	gpio->dscr_qspi		= 0x01;
  DeAssertCE();

  // 5 MHz Clock
  clockDiv = (UINT8)(HwGetSystemClockHz() / (2 * 5000000));
  qspi->qspi_mode_reg = 0xC000 | clockDiv;	
	qspi->qspi_wrap_reg = 0x1000;
	qspi->qspi_addr_reg = 0x20;
	qspi->qspi_data_reg = 0x8000;
	qspi->qspi_data_reg = 0x4000;
	qspi->qspi_int_reg  = 0xD00F;
	

  //gpio->par_qspi = 
  //  MCF_GPIO_PAR_QSPI_PAR_SCK(0x3) | 
  //  MCF_GPIO_PAR_QSPI_PAR_DOUT | 
  //  MCF_GPIO_PAR_QSPI_PAR_DIN(0x3) | 
  //  MCF_GPIO_PAR_QSPI_PAR_PCS0;


}

// MNT - 4/5/2007 - These are folded into the SPI driver now
#if 0

/*!
 * \def SPIWaitTransferDone()
 * \brief Waits until the SPI1D has been transferred and received
 */
#pragma INLINE
void SPIWaitTransferDone(void){
  while (!(SPI1S_SPRF));
}

/*!
 * 
 * \def SPIClearRecieveStatReg()
 * \brief SPI_ClearRecieveStatReg : Clears any flags that may be activated in the
 * the SPI Data Register.
 */
#pragma INLINE
void SPIClearRecieveStatReg(UINT8 * u8TempValue){
  *u8TempValue = SPI1S;
}

/*! 
 * \def SPIClearRecieveDataReg()
 * \brief SPIClearRecieveDataReg : Clear receive data register. SPI entirely ready 
 * for read or write 
 */
#pragma INLINE
void SPIClearRecieveDataReg(UINT8 * u8TempValue){
  *u8TempValue = SPI1D;
}

/*! 
 * \def SPISendChar()
 * \brief SPI_SendChar : Sends a Char through the SPI port that is being used
 *
 * Parameters : u8Char - Char to be sent through the SPI
 */
#pragma INLINE
void SPISendChar(UINT8 u8Char){
  SPI1D = u8Char;
}

/*! 
 * \def SPIRead()
 * \brief SPIRead : Returns the data that is in the SPI port.
 * Return : UINT8 - Data that is in the SPI port being used.
 */
#pragma INLINE
UINT8 SPIRead(void){
  return SPI1D;	 /*!<SPI Data Register*/
}


#pragma INLINE
void SPIReadMSB(UINT16 * u16Data){
  ((UINT8*)u16Data)[0] = SPI1D; /*!<SPI More Significant Byte*/	
//  ((UINT8*)u16Data)[0] = SPIRead(); /*!<SPI More Significant Byte*/
}

#pragma INLINE
void SPIReadLSB(UINT16 * u16Data){
  ((UINT8*)u16Data)[1] = SPI1D; /*!<SPI Less Significant Byte*/
//  ((UINT8*)u16Data)[1] = SPIRead(); /*!<SPI Less Significant Byte*/
}

#endif


/*!
 * \brief SPIDrvWrite : Write 1 word to SPI
 *
 * \param u8Addr - SPI address
 * \param u16Content - Data to send
 *
 */
void SPIDrvWrite(UINT8 addr, UINT16 content)
{
	/*disable interrupts to avoid double access to zigbee spi*/
	MC13192DisableInterrupts();
	/*Initialize Tx pkt Ram*/
	MCF_QSPI_QAR = 0x20;		/*points to the command ram*/
	MCF_QSPI_QDR = 0x0000;		/*initialize CR for 8 bit xfer*/
	MCF_QSPI_QAR = 0x00;		/*points to TX pkt ram*/
	MCF_QSPI_QDR = addr; 		/*loads address */
	AssertCE();
	MCF_QSPI_QDLYR = 0x8000; /*will tsfr above info to zigbee*/
	SPIWaitTransferDone ();
	MCF_QSPI_QIR = (0xD00F | MCF_QSPI_QIR_SPIF);
	MCF_QSPI_QAR = 0x20;		/*points to the command ram*/
	MCF_QSPI_QDR = 0x4000;		/*initialize CR for 16 bit xfer*/
	MCF_QSPI_QAR = 0x00;		/*points to TX pkt ram*/
	MCF_QSPI_QDR = content; 		/*loads high byte */
	MCF_QSPI_QDLYR = 0x8000; 	/*will tsfr above info to zigbee*/
	SPIWaitTransferDone ();
	MCF_QSPI_QIR = (0xD00F | MCF_QSPI_QIR_SPIF);
	DeAssertCE();
	/*enable interrupts*/
	MC13192RestoreInterrupts();
}


/*!
 * \brief SPIDrvRead : Read 1 word from SPI
 *
 * \param u8Addr - SPI address
 *
 * \return u16Data -  u16Data[0] is the MSB, u16Data[1] is the LSB
 */
UINT16 SPIDrvRead(UINT8 addr)
{
	UINT16 value;
	
	MC13192DisableInterrupts();
	/*Initialize Tx pkt Ram*/
	MCF_QSPI_QAR = 0x20;		/*points to the command ram*/
	MCF_QSPI_QDR = 0x0000;		/*initialize CR for pkt*/
	MCF_QSPI_QAR = 0x00;		/*points to TX pkt ram*/
	MCF_QSPI_QDR = (addr | 0x80); 		/*sets destination*/
	AssertCE();
	MCF_QSPI_QDLYR = 0x8000; /*will tsfr above info to zigbee*/
	SPIWaitTransferDone ();
	MCF_QSPI_QIR = (MCF_QSPI_QIR | MCF_QSPI_QIR_SPIF);
	MCF_QSPI_QAR = 0x20;		/*points to the command ram*/
	MCF_QSPI_QDR = 0x4000;		/*initialize CR for pkt*/
	MCF_QSPI_QDLYR = 0x8000; /*will tsfr above info to zigbee*/
	SPIWaitTransferDone ();
	MCF_QSPI_QIR = (MCF_QSPI_QIR | MCF_QSPI_QIR_SPIF);
	MCF_QSPI_QAR = 0x10;
	value = MCF_QSPI_QDR;
	DeAssertCE();
	MC13192RestoreInterrupts();
	return value;	
}

/*!
 * \brief SPIDrvRead2 : Read the second word from SPI
 *
 * \param u8Addr - SPI address
 *
 * \return u16Data -  u16Data[0] is the MSB, u16Data[1] is the LSB
 */
UINT16 SPIDrvRead2(UINT8 addr)
{
	UINT8 tempvalue;
	UINT16 value;
	
	MC13192DisableInterrupts();
	/*Initialize Tx pkt Ram*/
	MCF_QSPI_QAR = 0x20;		/*points to the command ram*/
	MCF_QSPI_QDR = 0x0000;		/*initialize CR for pkt*/
	MCF_QSPI_QAR = 0x00;		/*points to TX pkt ram*/
	MCF_QSPI_QDR = (addr | 0x80); 		/*sets destination*/
	AssertCE();
	MCF_QSPI_QDLYR = 0x8000; /*will tsfr above info to zigbee*/
	SPIWaitTransferDone ();
	MCF_QSPI_QIR = (MCF_QSPI_QIR | MCF_QSPI_QIR_SPIF);
	MCF_QSPI_QAR = 0x20;		/*points to the command ram*/
	MCF_QSPI_QDR = 0x4000;		/*initialize CR for pkt*/
	MCF_QSPI_QDLYR = 0x8000; /*will tsfr above info to zigbee*/
	SPIWaitTransferDone ();
	MCF_QSPI_QIR = (MCF_QSPI_QIR | MCF_QSPI_QIR_SPIF);
	MCF_QSPI_QDLYR = 0x8000; /*will tsfr above info to zigbee*/
	SPIWaitTransferDone ();
	MCF_QSPI_QIR = (MCF_QSPI_QIR | MCF_QSPI_QIR_SPIF);
	MCF_QSPI_QAR = 0x10;
	value = MCF_QSPI_QDR;
	DeAssertCE();
	MC13192RestoreInterrupts();
	return value;	
}


/*!
 * \brief RAMDrvWriteTx : Write a block of data to TX packet RAM, whichever is selected
 *
 * \param *psTxPkt - Packet to write
 *
 */
void RAMDrvWriteTx(tTxPacket *tx_pkt)
{
	UINT16 tmp;
	UINT8 tmp2;
	int i;
	
	/*Prepare the packet length by adding 2 to the total value*/ 
	/*for MDR purposes*/
	tmp = SPIDrvRead(TX_PKT_LEN);
  // MNT - 4/27/2007 - The Header will now be put in by the MAC layer above this
  // We only add for the CRC
	SPIDrvWrite(TX_PKT_LEN,((tmp & 0xFF80) | (tx_pkt->u8DataLength + 2) ));
	MC13192DisableInterrupts();
	MCF_QSPI_QAR = 0x20;
	MCF_QSPI_QDR = 0x0000;
	MCF_QSPI_QAR = 0x00;			/*points to TX pkt ram*/
	MCF_QSPI_QDR = TX_PKT;
	/*Drop chip select and send address and first byte*/
	AssertCE();
	MCF_QSPI_QDLYR = 0x8000; /*will tsfr above info to zigbee*/
	SPIWaitTransferDone ();
	MCF_QSPI_QIR = (0xD00F | MCF_QSPI_QIR_SPIF);
	MCF_QSPI_QAR = 0x20;		/*points to the command ram*/
	MCF_QSPI_QDR = 0x4000;	    /*16 bit transfers*/

  // MNT - 4/27/2007 - The Header will now be put in by the MAC layer above this
#if 0
  MCF_QSPI_QAR = 0x00;			/*points to TX pkt ram*/
	MCF_QSPI_QDR = 0xFF7E;
	MCF_QSPI_QDLYR = 0x8000; /*will tsfr above info to zigbee*/
	SPIWaitTransferDone ();
	MCF_QSPI_QIR = (0xD00F | MCF_QSPI_QIR_SPIF);
#endif // disable extra header byte write
  
  for(i=0; i<=tx_pkt->u8DataLength;)
	{
		MCF_QSPI_QAR = 0x00;
		MCF_QSPI_QDR = (((UINT16)tx_pkt->pu8Data[i+1])<<8|(UINT16)tx_pkt->pu8Data[i]);	/* high byte */
		MCF_QSPI_QDLYR = 0x8000; /*will tsfr above info to zigbee*/
		SPIWaitTransferDone ();
		MCF_QSPI_QIR = (0xD00F | MCF_QSPI_QIR_SPIF);
		i = i+2;
	}
	DeAssertCE();
	MC13192RestoreInterrupts();
}


/*!
 * \brief RAMDrvReadRx : Read a block of data from RX packet RAM, whichever is selected
 *
 * \param *psRxPkt - Packet received
 *
 * \return u8Status
 */
UINT8 RAMDrvReadRx(tRxPacket *rx_pkt)
{
	UINT16 tmp;
	UINT8 tmp2;

	int i;	

  /* MC13192 reports length with 2 CRC bytes
   * remove them. 
   */
	rx_pkt->u8DataLength  = rx_pkt->u8DataLength  - 2;
	
	MC13192DisableInterrupts();	
  	MCF_QSPI_QAR = 0x20;		/*points to the command ram*/
	MCF_QSPI_QDR = 0x0000;		/*initialize CR for pkt*/
  	/*Grab first byte*/
  	MCF_QSPI_QAR = 0x00;		/*points to TX pkt ram*/
	MCF_QSPI_QDR = 0x81; 		/*sets destination*/
  	AssertCE();
	MCF_QSPI_QDLYR = 0x8000; /*will tsfr above info to zigbee*/
	SPIWaitTransferDone ();
  	MCF_QSPI_QIR = (MCF_QSPI_QIR | MCF_QSPI_QIR_SPIF); /*clears tsfr status bit*/
  	MCF_QSPI_QAR = 0x20;		/*points to the command ram*/
	MCF_QSPI_QDR = 0x4000;		/*initialize CR for pkt*/
	MCF_QSPI_QDLYR = 0x8000; /*will tsfr above info to zigbee*/
	SPIWaitTransferDone ();
  	MCF_QSPI_QIR = (MCF_QSPI_QIR | MCF_QSPI_QIR_SPIF); /*clears tsfr status bit*/
	/*2 bytes have been ignored*/

    // MNT - 4/27/2007 - We only need to ignore the first 2 bytes based on the
    // datasheet. 
#if 0
	MCF_QSPI_QDLYR = 0x8000; /*will tsfr above info to zigbee*/
  SPIWaitTransferDone ();
  	MCF_QSPI_QIR = (MCF_QSPI_QIR | MCF_QSPI_QIR_SPIF); /*clears tsfr status bit*/
	/*2byte codes have been ignored*/
#endif 

	for(i = 0; i<= rx_pkt->u8DataLength ;)
  	{
  		MCF_QSPI_QDLYR = 0x8000; /*will tsfr above info to zigbee*/
		SPIWaitTransferDone ();
  		MCF_QSPI_QIR = (MCF_QSPI_QIR | MCF_QSPI_QIR_SPIF); /*clears tsfr status bit*/
		MCF_QSPI_QAR = 0x10;
		tmp = MCF_QSPI_QDR;
		if(i+1 == rx_pkt->u8DataLength )
		{
			/*do nothing garbage byte*/
		}
		else
		{
			rx_pkt->pu8Data[i+1] = (tmp>>8);
		}
		rx_pkt->pu8Data[i] = tmp;
		i = i + 2;
	}
	DeAssertCE();
	MC13192RestoreInterrupts();
	return SMAC_SUCCESS;
}




#pragma inline SPIWaitTransferDone
void SPIWaitTransferDone(void) 
{
  while (!(MCF_QSPI_QIR & MCF_QSPI_QIR_SPIF));
}
